import express from "express";

const app = express();

const PORT = Number(process.env.PORT || 3005);
const UPSTREAM_BASE = process.env.UPSTREAM_BASE || "https://themoviebox.org/wefeed-h5api-bff";
const CACHE_TTL_MS = Number(process.env.CACHE_TTL_MS || 60000);
const ALLOW_ORIGIN = process.env.ALLOW_ORIGIN || "*";

const cache = new Map();

function setCache(key, value, ttlMs = CACHE_TTL_MS) {
  cache.set(key, { value, expiresAt: Date.now() + ttlMs });
}

function getCache(key) {
  const hit = cache.get(key);
  if (!hit) return null;
  if (hit.expiresAt < Date.now()) {
    cache.delete(key);
    return null;
  }
  return hit.value;
}

function baseHeaders() {
  return {
    "User-Agent": "Mozilla/5.0 (Windows NT 10.0; Win64; x64)",
    "Referer": "https://themoviebox.org/",
    "Origin": "https://themoviebox.org",
    "Accept": "application/json, text/plain, */*",
    "Accept-Language": "id-ID,id;q=0.9,en-US;q=0.8,en;q=0.7"
  };
}

async function fetchJson(url) {
  const response = await fetch(url, {
    headers: baseHeaders(),
    cache: "no-store"
  });

  if (!response.ok) {
    const text = await response.text().catch(() => "");
    const error = new Error("Upstream error");
    error.status = response.status;
    error.body = text;
    throw error;
  }

  return response.json();
}

app.use((req, res, next) => {
  res.setHeader("Access-Control-Allow-Origin", ALLOW_ORIGIN);
  res.setHeader("Access-Control-Allow-Methods", "GET, OPTIONS");
  res.setHeader("Access-Control-Allow-Headers", "Content-Type");
  if (req.method === "OPTIONS") return res.sendStatus(204);
  next();
});

app.get("/moviebox/homepage", async (req, res) => {
  const cacheKey = "homepage";
  const cached = getCache(cacheKey);
  if (cached) return res.json(cached);

  try {
    const data = await fetchJson(`${UPSTREAM_BASE}/home/homepage`);
    setCache(cacheKey, data, CACHE_TTL_MS);
    res.json(data);
  } catch (error) {
    res.status(error.status || 500).json({ error: "Failed to fetch homepage" });
  }
});

app.get("/moviebox/search", async (req, res) => {
  const query = (req.query.query || "").toString().trim();
  if (!query) return res.status(400).json({ error: "query is required" });

  const cacheKey = `search:${query}`;
  const cached = getCache(cacheKey);
  if (cached) return res.json(cached);

  try {
    const url = `${UPSTREAM_BASE}/search/search?keyword=${encodeURIComponent(query)}&page=1&size=20`;
    const data = await fetchJson(url);
    setCache(cacheKey, data, 30 * 1000);
    res.json(data);
  } catch (error) {
    res.status(error.status || 500).json({ error: "Failed to fetch search" });
  }
});

app.get("/moviebox/detail", async (req, res) => {
  const subjectId = (req.query.subjectId || "").toString().trim();
  if (!subjectId) return res.status(400).json({ error: "subjectId is required" });

  const cacheKey = `detail:${subjectId}`;
  const cached = getCache(cacheKey);
  if (cached) return res.json(cached);

  try {
    const url = `${UPSTREAM_BASE}/subject/detail?subjectId=${encodeURIComponent(subjectId)}`;
    const data = await fetchJson(url);
    setCache(cacheKey, data, 5 * 60 * 1000);
    res.json(data);
  } catch (error) {
    res.status(error.status || 500).json({ error: "Failed to fetch detail" });
  }
});

app.get("/moviebox/play", async (req, res) => {
  const subjectId = (req.query.subjectId || "").toString().trim();
  const detailPath = (req.query.detailPath || "").toString().trim();
  const se = (req.query.se || "1").toString().trim();
  const ep = (req.query.ep || "1").toString().trim();

  if (!subjectId || !detailPath) {
    return res.status(400).json({ error: "subjectId and detailPath are required" });
  }

  const cacheKey = `play:${subjectId}:${detailPath}:${se}:${ep}`;
  const cached = getCache(cacheKey);
  if (cached) return res.json(cached);

  try {
    const url = `${UPSTREAM_BASE}/subject/play?subjectId=${encodeURIComponent(subjectId)}&se=${encodeURIComponent(se)}&ep=${encodeURIComponent(ep)}&detailPath=${encodeURIComponent(detailPath)}`;
    const data = await fetchJson(url);
    setCache(cacheKey, data, 30 * 1000);
    res.json(data);
  } catch (error) {
    res.status(error.status || 500).json({ error: "Failed to fetch play" });
  }
});

app.get("/moviebox/sources", async (req, res) => {
  const subjectId = (req.query.subjectId || "").toString().trim();
  if (!subjectId) return res.status(400).json({ error: "subjectId is required" });

  const cacheKey = `sources:${subjectId}`;
  const cached = getCache(cacheKey);
  if (cached) return res.json(cached);

  try {
    const url = `${UPSTREAM_BASE}/subject/source?subjectId=${encodeURIComponent(subjectId)}`;
    const data = await fetchJson(url);
    setCache(cacheKey, data, 60 * 1000);
    res.json(data);
  } catch (error) {
    res.status(error.status || 500).json({ error: "Failed to fetch sources" });
  }
});

app.get("/health", (req, res) => {
  res.json({ ok: true });
});

app.listen(PORT, () => {
  console.log(`MovieBox API listening on ${PORT}`);
});
